<?php

namespace App\Traits;

use Exception;
use App\Models\User;
use App\Enums\SettingKey;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Enums\MetaApiEndpoints;
use Illuminate\Support\Facades\Http;

trait MetaApiTrait
{
     protected string $metaApiBaseUrl = 'https://www.facebook.com';
     protected string $metaApiGraphUrl = 'https://graph.facebook.com';
     protected string $defaultMetaApiVersion = 'v21.0';

     public function makeMetaApiRequest(MetaApiEndpoints|string $endpoint, array $params = [], string $method = 'get', ?string $version = null): array
     {
          try {
               $endpointValue = $endpoint instanceof MetaApiEndpoints ? $endpoint->value : $endpoint; 
               $version = $version ?? $this->defaultMetaApiVersion;
               $url = $endpoint === MetaApiEndpoints::OAUTH_ACCESS_TOKEN
                    ? "{$this->metaApiGraphUrl}" . $this->buildEndpoint($endpointValue, $params)
                    : "{$this->metaApiGraphUrl}/{$version}/" . $this->buildEndpoint($endpointValue, $params);
               
               $http = Http::withHeaders([]);
               if ($method === 'post') {
                    $http = $http->asForm();
               }
               if ($endpoint !== MetaApiEndpoints::OAUTH_ACCESS_TOKEN) {
                    $http = $http->withHeaders(
                         collect($params)->mapWithKeys(function ($value, $key) {
                              return $key === 'access_token' ? ['Authorization' => "Bearer $value"] : [];
                         })->all()
                    );
               }
               // 'Content-Type' => 'application/json',

               $response = $http->$method($url, $params);

               if (!$response->successful()) {
                    $error = Arr::get($response->json(), 'error', []);
                    return [
                         'success' => false,
                         'message' => Arr::get($error, 'message', 'Unknown error'),
                         'error' => $error
                    ];
               }

               $data = $response->json();
               if ($endpoint === MetaApiEndpoints::OAUTH_ACCESS_TOKEN && !Arr::has($data, 'access_token')) {
                    return [
                         'success' => false,
                         'message' => 'No access token in response',
                         'error' => $data
                    ];
               }

               return [
                    'success' => true,
                    'data' => $data
               ];
          } catch (Exception $e) {
               
               return [
                    'success' => false,
                    'message' => $e->getMessage()
               ];
          }
     }

     private function buildEndpoint(string $endpoint, array &$params): string
     {
          return collect(explode('/', $endpoint))->map(function ($segment) use (&$params) {
               if (str_starts_with($segment, ':')) {
                    $key = ltrim($segment, ':');
                    $value = Arr::get($params, $key);
                    Arr::forget($params, $key);
                    return $value ?? $segment;
               }
               return $segment;
          })->implode('/');
     }

     public function initiateMetaEmbeddedSignup(Request $request, string $redirectRoute, array $scopes = ['whatsapp_business_messaging', 'business_management'], string $feature = 'whatsapp_embedded_signup', string $flow = 'signup', User|null $user = null): array
     {
          try {
               $appId = site_settings(SettingKey::META_APP_ID->value);
               $appSecret = site_settings(SettingKey::META_APP_SECRET->value);

               if (empty($appId) || empty($appSecret)) {
                    return [
                         'success' => false,
                         'message' => translate('Meta App credentials not configured. Please configure App ID and App Secret in settings.')
                    ];
               }

               $state = base64_encode(json_encode([
                    'user_type' => $user ? "user" : 'admin',
                    'timestamp' => now()->timestamp,
                    'nonce' => Str::random(16),
               ]));

               $signupUrl = "{$this->metaApiBaseUrl}/{$this->defaultMetaApiVersion}/dialog/oauth?" . http_build_query([
                    'client_id' => $appId,
                    'redirect_uri' => route($redirectRoute),
                    'state' => $state,
                    'scope' => implode(',', $scopes),
                    'extras' => json_encode([
                         'feature' => $feature,
                         'version' => 2,
                         'setup' => [
                              'solution' => 'whatsapp',
                              'flow' => $flow
                         ]
                    ])
               ]);

               return [
                    'success' => true,
                    'signup_url' => $signupUrl,
                    'message' => translate('Embedded signup URL generated successfully')
               ];
          } catch (Exception $e) {
               return [
                    'success' => false,
                    'message' => translate('Failed to initiate embedded signup: ') . $e->getMessage()
               ];
          }
     }
}