@push("style-include")
<link rel="stylesheet" href="{{ asset('assets/theme/global/css/select2.min.css') }}">
<style>
/* Health Status Cards */
.health-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}
.health-card {
    background: var(--card-bg, #fff);
    border-radius: 12px;
    padding: 1.25rem;
    border: 1px solid var(--border-color, #e5e7eb);
    display: flex;
    align-items: center;
    gap: 1rem;
}
.health-icon {
    width: 48px;
    height: 48px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
}
.health-icon.success { background: rgba(16, 185, 129, 0.1); color: #10b981; }
.health-icon.warning { background: rgba(245, 158, 11, 0.1); color: #f59e0b; }
.health-icon.danger { background: rgba(239, 68, 68, 0.1); color: #ef4444; }
.health-icon.info { background: rgba(59, 130, 246, 0.1); color: #3b82f6; }
.health-label { font-size: 0.75rem; color: var(--text-muted, #6b7280); margin-bottom: 0.25rem; }
.health-value { font-size: 1.25rem; font-weight: 600; margin: 0; }

/* Server Type Selection */
.server-types {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin: 1.5rem 0;
}
.server-type-card {
    background: var(--card-bg, #fff);
    border: 2px solid var(--border-color, #e5e7eb);
    border-radius: 16px;
    padding: 1.5rem;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}
.server-type-card:hover {
    border-color: var(--primary-color, #6366f1);
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.15);
}
.server-type-card.selected {
    border-color: var(--primary-color, #6366f1);
    background: rgba(99, 102, 241, 0.05);
}
.server-type-card.selected::after {
    content: '\f00c';
    font-family: 'remixicon';
    position: absolute;
    top: 1rem;
    right: 1rem;
    width: 24px;
    height: 24px;
    background: var(--primary-color, #6366f1);
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.75rem;
}
.server-type-icon {
    width: 56px;
    height: 56px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.75rem;
    margin-bottom: 1rem;
}
.server-type-icon.cpanel { background: linear-gradient(135deg, #ff6a00 0%, #ee0979 100%); color: white; }
.server-type-icon.vps { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
.server-type-icon.dedicated { background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%); color: white; }
.server-type-title { font-size: 1.125rem; font-weight: 600; margin-bottom: 0.5rem; }
.server-type-desc { font-size: 0.875rem; color: var(--text-muted, #6b7280); margin-bottom: 1rem; }
.server-type-features { list-style: none; padding: 0; margin: 0; }
.server-type-features li {
    font-size: 0.8125rem;
    color: var(--text-muted, #6b7280);
    padding: 0.25rem 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}
.server-type-features li i { color: #10b981; }

/* Setup Steps */
.setup-container { display: none; }
.setup-container.active { display: block; }
.setup-steps {
    counter-reset: step;
    margin: 0;
    padding: 0;
    list-style: none;
}
.setup-step {
    position: relative;
    padding: 1.5rem;
    padding-left: 4rem;
    border-left: 2px solid var(--border-color, #e5e7eb);
    margin-left: 1rem;
}
.setup-step:last-child { border-left-color: transparent; }
.setup-step::before {
    counter-increment: step;
    content: counter(step);
    position: absolute;
    left: -1rem;
    top: 1.5rem;
    width: 2rem;
    height: 2rem;
    background: var(--primary-color, #6366f1);
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    font-size: 0.875rem;
}
.step-title { font-size: 1rem; font-weight: 600; margin-bottom: 0.5rem; }
.step-desc { font-size: 0.875rem; color: var(--text-muted, #6b7280); margin-bottom: 1rem; }

/* Code Block */
.code-box {
    background: #1e293b;
    border-radius: 8px;
    padding: 1rem;
    position: relative;
    margin: 0.5rem 0;
}
.code-box code {
    color: #e2e8f0;
    font-family: 'Monaco', 'Menlo', monospace;
    font-size: 0.8125rem;
    word-break: break-all;
    display: block;
    padding-right: 3rem;
}
.code-box .copy-btn {
    position: absolute;
    top: 0.5rem;
    right: 0.5rem;
    background: rgba(255,255,255,0.1);
    border: none;
    color: #94a3b8;
    padding: 0.5rem;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s;
}
.code-box .copy-btn:hover { background: rgba(255,255,255,0.2); color: white; }

/* Alert boxes */
.tip-box {
    background: rgba(59, 130, 246, 0.1);
    border: 1px solid rgba(59, 130, 246, 0.2);
    border-radius: 8px;
    padding: 1rem;
    margin: 1rem 0;
    display: flex;
    gap: 0.75rem;
}
.tip-box i { color: #3b82f6; font-size: 1.25rem; flex-shrink: 0; }
.tip-box p { margin: 0; font-size: 0.875rem; color: var(--text-color); }

.warning-box {
    background: rgba(245, 158, 11, 0.1);
    border: 1px solid rgba(245, 158, 11, 0.2);
    border-radius: 8px;
    padding: 1rem;
    margin: 1rem 0;
    display: flex;
    gap: 0.75rem;
}
.warning-box i { color: #f59e0b; font-size: 1.25rem; flex-shrink: 0; }
.warning-box p { margin: 0; font-size: 0.875rem; color: var(--text-color); }

/* Quick Action Buttons */
.quick-actions {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
    margin-top: 1rem;
}
</style>
@endpush

@extends('admin.layouts.app')
@section("panel")
<main class="main-body">
    <div class="container-fluid px-0 main-content">
        <div class="page-header">
            <div class="page-header-left">
                <h2 class="page-title">{{ translate('Automation Settings') }}</h2>
                <div class="breadcrumb-wrapper">
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item">
                                <a href="{{ route('admin.dashboard') }}">{{ translate("Dashboard") }}</a>
                            </li>
                            <li class="breadcrumb-item active" aria-current="page">{{ translate('Automation') }}</li>
                        </ol>
                    </nav>
                </div>
            </div>
        </div>

        <!-- System Health Status -->
        <div class="card mb-4">
            <div class="card-header">
                <div class="card-header-left">
                    <h4 class="card-title">
                        <i class="ri-pulse-line"></i>
                        {{ translate("System Health") }}
                    </h4>
                </div>
                <div class="card-header-right">
                    <button type="button" class="i-btn btn--outline btn--sm" onclick="refreshHealth()">
                        <i class="ri-refresh-line"></i> {{ translate("Refresh") }}
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="health-grid">
                    <div class="health-card">
                        <div class="health-icon" id="healthStatusIcon">
                            <i class="ri-loader-4-line"></i>
                        </div>
                        <div>
                            <div class="health-label">{{ translate("Status") }}</div>
                            <p class="health-value" id="healthStatus">{{ translate("Checking...") }}</p>
                        </div>
                    </div>
                    <div class="health-card">
                        <div class="health-icon info">
                            <i class="ri-time-line"></i>
                        </div>
                        <div>
                            <div class="health-label">{{ translate("Last Cron Run") }}</div>
                            <p class="health-value" id="lastCronRun">-</p>
                        </div>
                    </div>
                    <div class="health-card">
                        <div class="health-icon warning">
                            <i class="ri-stack-line"></i>
                        </div>
                        <div>
                            <div class="health-label">{{ translate("Pending Jobs") }}</div>
                            <p class="health-value" id="pendingJobs">-</p>
                        </div>
                    </div>
                    <div class="health-card">
                        <div class="health-icon danger">
                            <i class="ri-error-warning-line"></i>
                        </div>
                        <div>
                            <div class="health-label">{{ translate("Failed Jobs") }}</div>
                            <p class="health-value" id="failedJobs">-</p>
                        </div>
                    </div>
                </div>

                <div id="healthWarnings" class="warning-box mt-3" style="display: none;">
                    <i class="ri-alert-line"></i>
                    <p id="healthWarningsText"></p>
                </div>

                <div class="quick-actions">
                    <button type="button" class="i-btn btn--success btn--sm" data-bs-toggle="modal" data-bs-target="#retryFailedModal">
                        <i class="ri-restart-line"></i> {{ translate("Retry Failed Jobs") }}
                    </button>
                    <button type="button" class="i-btn btn--danger btn--sm" data-bs-toggle="modal" data-bs-target="#clearFailedModal">
                        <i class="ri-delete-bin-line"></i> {{ translate("Clear Failed Jobs") }}
                    </button>
                    <button type="button" class="i-btn btn--primary btn--sm" data-bs-toggle="modal" data-bs-target="#queueConnectionModal">
                        <i class="ri-settings-3-line"></i> {{ translate("Queue Settings") }}
                    </button>
                </div>
            </div>
        </div>

        <!-- Automation Mode Selection -->
        <div class="card mb-4">
            <div class="card-header">
                <div class="card-header-left">
                    <h4 class="card-title">
                        <i class="ri-settings-4-line"></i>
                        {{ translate("Automation Mode") }}
                    </h4>
                </div>
                <div class="card-header-right">
                    <span class="badge bg-{{ $currentMode === 'auto' ? 'info' : ($currentMode === 'supervisor' ? 'success' : 'primary') }}">
                        {{ translate("Current:") }} {{ ucfirst(str_replace('_', ' ', $currentMode ?? 'auto')) }}
                    </span>
                </div>
            </div>
            <div class="card-body">
                <div class="warning-box mb-3">
                    <i class="ri-information-line"></i>
                    <p>
                        <strong>{{ translate("Important:") }}</strong> {{ translate("Select the mode that matches your setup to prevent double message sending. If unsure, use 'Auto Detect'.") }}
                    </p>
                </div>

                <form id="automationModeForm" class="mb-4">
                    @csrf
                    <div class="row g-3 align-items-end">
                        <div class="col-md-8">
                            <label class="form-label">{{ translate("Select Automation Mode") }}</label>
                            <select name="automation_mode" id="automationModeSelect" class="form-select">
                                <option value="auto" {{ ($currentMode ?? 'auto') === 'auto' ? 'selected' : '' }}>
                                    🔮 {{ translate("Auto Detect (Recommended)") }}
                                </option>
                                <option value="cron_url" {{ ($currentMode ?? '') === 'cron_url' ? 'selected' : '' }}>
                                    ☁️ {{ translate("Cron URL Only (cPanel/Shared Hosting)") }}
                                </option>
                                <option value="scheduler" {{ ($currentMode ?? '') === 'scheduler' ? 'selected' : '' }}>
                                    💻 {{ translate("Laravel Scheduler (VPS)") }}
                                </option>
                                <option value="supervisor" {{ ($currentMode ?? '') === 'supervisor' ? 'selected' : '' }}>
                                    🏢 {{ translate("Supervisor Workers (Enterprise)") }}
                                </option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <button type="submit" class="i-btn btn--primary btn--md w-100">
                                <i class="ri-save-line"></i> {{ translate("Save Mode") }}
                            </button>
                        </div>
                    </div>
                </form>

                <div id="modeDescription" class="tip-box">
                    <i class="ri-lightbulb-line"></i>
                    <p id="modeDescriptionText">{{ translate("Auto Detect: System automatically detects supervisor workers and adjusts behavior.") }}</p>
                </div>

                <div class="mt-3 p-3 bg-light rounded">
                    <h6 class="mb-2"><i class="ri-question-line"></i> {{ translate("Which mode should I use?") }}</h6>
                    <ul class="small text-muted mb-0">
                        <li><strong>{{ translate("Auto Detect:") }}</strong> {{ translate("Best for most users. System decides based on your setup.") }}</li>
                        <li><strong>{{ translate("Cron URL:") }}</strong> {{ translate("For cPanel/shared hosting. Use /automation/run URL in cron job.") }}</li>
                        <li><strong>{{ translate("Scheduler:") }}</strong> {{ translate("For VPS with 'artisan schedule:run' cron. No supervisor.") }}</li>
                        <li><strong>{{ translate("Supervisor:") }}</strong> {{ translate("For enterprise with supervisor running queue:work continuously.") }}</li>
                    </ul>
                </div>
            </div>
        </div>

        <!-- Server Type Selection -->
        <div class="card mb-4">
            <div class="card-header">
                <div class="card-header-left">
                    <h4 class="card-title">
                        <i class="ri-server-line"></i>
                        {{ translate("Setup Instructions") }}
                    </h4>
                </div>
            </div>
            <div class="card-body">
                <p class="text-muted mb-3">{{ translate("Choose your hosting type to see the appropriate setup instructions.") }}</p>

                <div class="server-types">
                    <!-- Shared Hosting / cPanel -->
                    <div class="server-type-card" data-server="cpanel" onclick="selectServerType('cpanel')">
                        <div class="server-type-icon cpanel">
                            <i class="ri-cloud-line"></i>
                        </div>
                        <h5 class="server-type-title">{{ translate("Shared Hosting") }}</h5>
                        <p class="server-type-desc">{{ translate("cPanel, Plesk, or any web hosting panel. Single URL setup.") }}</p>
                        <ul class="server-type-features">
                            <li><i class="ri-check-line"></i> {{ translate("No terminal required") }}</li>
                            <li><i class="ri-check-line"></i> {{ translate("Web-based cron setup") }}</li>
                            <li><i class="ri-check-line"></i> {{ translate("Beginner friendly") }}</li>
                        </ul>
                    </div>

                    <!-- VPS / Local Development -->
                    <div class="server-type-card" data-server="vps" onclick="selectServerType('vps')">
                        <div class="server-type-icon vps">
                            <i class="ri-computer-line"></i>
                        </div>
                        <h5 class="server-type-title">{{ translate("VPS / Local Server") }}</h5>
                        <p class="server-type-desc">{{ translate("Linux, Windows, or macOS with terminal access.") }}</p>
                        <ul class="server-type-features">
                            <li><i class="ri-check-line"></i> {{ translate("All operating systems") }}</li>
                            <li><i class="ri-check-line"></i> {{ translate("Laravel scheduler") }}</li>
                            <li><i class="ri-check-line"></i> {{ translate("Better performance") }}</li>
                        </ul>
                    </div>

                    <!-- Dedicated / Enterprise -->
                    <div class="server-type-card" data-server="dedicated" onclick="selectServerType('dedicated')">
                        <div class="server-type-icon dedicated">
                            <i class="ri-database-2-line"></i>
                        </div>
                        <h5 class="server-type-title">{{ translate("Enterprise / High Volume") }}</h5>
                        <p class="server-type-desc">{{ translate("For high-volume sending with process management.") }}</p>
                        <ul class="server-type-features">
                            <li><i class="ri-check-line"></i> {{ translate("Process supervisor") }}</li>
                            <li><i class="ri-check-line"></i> {{ translate("Maximum throughput") }}</li>
                            <li><i class="ri-check-line"></i> {{ translate("Auto-restart on failure") }}</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>

        <!-- cPanel Setup Instructions -->
        <div class="setup-container" id="setup-cpanel">
            <div class="card">
                <div class="card-header">
                    <div class="card-header-left">
                        <h4 class="card-title">
                            <i class="ri-cloud-line text-primary"></i>
                            {{ translate("Shared Hosting / cPanel Setup") }}
                        </h4>
                    </div>
                </div>
                <div class="card-body">
                    <div class="tip-box">
                        <i class="ri-lightbulb-line"></i>
                        <p>{{ translate("This setup uses a single URL that handles everything - campaigns, messages, and queue processing. Just add one cron job and you're done!") }}</p>
                    </div>

                    <ol class="setup-steps">
                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Login to cPanel") }}</h5>
                            <p class="step-desc">{{ translate("Access your hosting control panel (usually at yourdomain.com/cpanel or yourdomain.com:2083)") }}</p>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Find Cron Jobs") }}</h5>
                            <p class="step-desc">{{ translate("Look for 'Cron Jobs' or 'Scheduled Tasks' in your cPanel. It's usually under 'Advanced' section.") }}</p>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Set Schedule to Every Minute") }}</h5>
                            <p class="step-desc">{{ translate("Select 'Once Per Minute' from the Common Settings dropdown, or enter these values:") }}</p>
                            <div class="code-box">
                                <code>* * * * *</code>
                                <button type="button" class="copy-btn" data-clipboard-text="* * * * *">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Add Command") }}</h5>
                            <p class="step-desc">{{ translate("Copy and paste this command into the 'Command' field:") }}</p>
                            <div class="code-box">
                                <code>curl -s {{ url('/automation/run') }} > /dev/null 2>&1</code>
                                <button type="button" class="copy-btn" data-clipboard-text="curl -s {{ url('/automation/run') }} > /dev/null 2>&1">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                            <p class="step-desc mt-2">{{ translate("Or if curl doesn't work, try wget:") }}</p>
                            <div class="code-box">
                                <code>wget -q -O /dev/null {{ url('/automation/run') }}</code>
                                <button type="button" class="copy-btn" data-clipboard-text="wget -q -O /dev/null {{ url('/automation/run') }}">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Save & Verify") }}</h5>
                            <p class="step-desc">{{ translate("Click 'Add New Cron Job'. Wait 2 minutes, then check 'Last Cron Run' above to verify it's working.") }}</p>
                        </li>
                    </ol>

                    <div class="warning-box">
                        <i class="ri-information-line"></i>
                        <p>{{ translate("Some shared hosts limit cron frequency. If every minute doesn't work, try every 5 minutes: */5 * * * *") }}</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- VPS / Local Server Setup Instructions -->
        <div class="setup-container" id="setup-vps">
            <div class="card">
                <div class="card-header">
                    <div class="card-header-left">
                        <h4 class="card-title">
                            <i class="ri-computer-line text-primary"></i>
                            {{ translate("VPS / Local Server Setup") }}
                        </h4>
                    </div>
                </div>
                <div class="card-body">
                    <div class="tip-box">
                        <i class="ri-lightbulb-line"></i>
                        <p>{{ translate("This setup uses Laravel's built-in scheduler for efficient task processing. Select your operating system below.") }}</p>
                    </div>

                    <!-- OS Tabs -->
                    <ul class="nav nav-tabs mb-4" id="osTab" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="linux-tab" data-bs-toggle="tab" data-bs-target="#linux-content" type="button" role="tab">
                                <i class="ri-ubuntu-line"></i> {{ translate("Linux") }}
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="windows-tab" data-bs-toggle="tab" data-bs-target="#windows-content" type="button" role="tab">
                                <i class="ri-windows-line"></i> {{ translate("Windows") }}
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="macos-tab" data-bs-toggle="tab" data-bs-target="#macos-content" type="button" role="tab">
                                <i class="ri-apple-line"></i> {{ translate("macOS") }}
                            </button>
                        </li>
                    </ul>

                    <div class="tab-content" id="osTabContent">
                        <!-- Linux Instructions -->
                        <div class="tab-pane fade show active" id="linux-content" role="tabpanel">
                            <ol class="setup-steps">
                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Open Terminal") }}</h5>
                                    <p class="step-desc">{{ translate("Connect to your server via SSH or open Terminal directly.") }}</p>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Edit Crontab") }}</h5>
                                    <p class="step-desc">{{ translate("Open the crontab editor:") }}</p>
                                    <div class="code-box">
                                        <code>crontab -e</code>
                                        <button type="button" class="copy-btn" data-clipboard-text="crontab -e">
                                            <i class="ri-file-copy-line"></i>
                                        </button>
                                    </div>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Add Scheduler Command") }}</h5>
                                    <p class="step-desc">{{ translate("Add this line at the bottom and save:") }}</p>
                                    <div class="code-box">
                                        <code>* * * * * cd {{ base_path() }} && php artisan schedule:run >> /dev/null 2>&1</code>
                                        <button type="button" class="copy-btn" data-clipboard-text="* * * * * cd {{ base_path() }} && php artisan schedule:run >> /dev/null 2>&1">
                                            <i class="ri-file-copy-line"></i>
                                        </button>
                                    </div>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Verify") }}</h5>
                                    <p class="step-desc">{{ translate("Save with Ctrl+X, Y, Enter (nano) or :wq (vim). Check health status after 2 minutes.") }}</p>
                                </li>
                            </ol>
                        </div>

                        <!-- Windows Instructions -->
                        <div class="tab-pane fade" id="windows-content" role="tabpanel">
                            <ol class="setup-steps">
                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Open Task Scheduler") }}</h5>
                                    <p class="step-desc">{{ translate("Press Win+R, type 'taskschd.msc' and press Enter. Or search 'Task Scheduler' in Start menu.") }}</p>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Create Basic Task") }}</h5>
                                    <p class="step-desc">{{ translate("Click 'Create Basic Task' in the right panel. Name it 'xSender Scheduler'.") }}</p>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Set Trigger") }}</h5>
                                    <p class="step-desc">{{ translate("Select 'Daily', set to repeat every 1 minute. In Advanced settings, check 'Repeat task every' and set to 1 minute.") }}</p>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Set Action") }}</h5>
                                    <p class="step-desc">{{ translate("Select 'Start a program'. Create a batch file with this content:") }}</p>
                                    <div class="code-box">
                                        <code>cd /d {{ base_path() }} && php artisan schedule:run</code>
                                        <button type="button" class="copy-btn" data-clipboard-text="cd /d {{ base_path() }} && php artisan schedule:run">
                                            <i class="ri-file-copy-line"></i>
                                        </button>
                                    </div>
                                    <p class="step-desc mt-2">{{ translate("Or run this PowerShell command to create the scheduled task:") }}</p>
                                    <div class="code-box">
                                        <code>$action = New-ScheduledTaskAction -Execute "php" -Argument "artisan schedule:run" -WorkingDirectory "{{ base_path() }}"
$trigger = New-ScheduledTaskTrigger -Once -At (Get-Date) -RepetitionInterval (New-TimeSpan -Minutes 1) -RepetitionDuration ([TimeSpan]::MaxValue)
Register-ScheduledTask -TaskName "xSenderScheduler" -Action $action -Trigger $trigger -RunLevel Highest</code>
                                        <button type="button" class="copy-btn" data-clipboard-text="$action = New-ScheduledTaskAction -Execute &quot;php&quot; -Argument &quot;artisan schedule:run&quot; -WorkingDirectory &quot;{{ base_path() }}&quot;; $trigger = New-ScheduledTaskTrigger -Once -At (Get-Date) -RepetitionInterval (New-TimeSpan -Minutes 1) -RepetitionDuration ([TimeSpan]::MaxValue); Register-ScheduledTask -TaskName &quot;xSenderScheduler&quot; -Action $action -Trigger $trigger -RunLevel Highest">
                                            <i class="ri-file-copy-line"></i>
                                        </button>
                                    </div>
                                </li>
                            </ol>
                        </div>

                        <!-- macOS Instructions -->
                        <div class="tab-pane fade" id="macos-content" role="tabpanel">
                            <ol class="setup-steps">
                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Open Terminal") }}</h5>
                                    <p class="step-desc">{{ translate("Open Terminal from Applications > Utilities, or press Cmd+Space and search 'Terminal'.") }}</p>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Edit Crontab") }}</h5>
                                    <p class="step-desc">{{ translate("Open the crontab editor:") }}</p>
                                    <div class="code-box">
                                        <code>crontab -e</code>
                                        <button type="button" class="copy-btn" data-clipboard-text="crontab -e">
                                            <i class="ri-file-copy-line"></i>
                                        </button>
                                    </div>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Add Scheduler Command") }}</h5>
                                    <p class="step-desc">{{ translate("Press 'i' to enter insert mode, add this line:") }}</p>
                                    <div class="code-box">
                                        <code>* * * * * cd {{ base_path() }} && php artisan schedule:run >> /dev/null 2>&1</code>
                                        <button type="button" class="copy-btn" data-clipboard-text="* * * * * cd {{ base_path() }} && php artisan schedule:run >> /dev/null 2>&1">
                                            <i class="ri-file-copy-line"></i>
                                        </button>
                                    </div>
                                </li>

                                <li class="setup-step">
                                    <h5 class="step-title">{{ translate("Save & Exit") }}</h5>
                                    <p class="step-desc">{{ translate("Press Esc, then type ':wq' and press Enter to save. Check health status after 2 minutes.") }}</p>
                                </li>
                            </ol>

                            <div class="warning-box">
                                <i class="ri-information-line"></i>
                                <p>{{ translate("On macOS, you may need to grant Terminal 'Full Disk Access' in System Preferences > Security & Privacy > Privacy.") }}</p>
                            </div>
                        </div>
                    </div>

                    <div class="tip-box mt-4">
                        <i class="ri-shield-check-line"></i>
                        <p>{{ translate("This method runs locally without exposing public URLs, making it more secure and efficient.") }}</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Enterprise / High Volume Setup Instructions -->
        <div class="setup-container" id="setup-dedicated">
            <div class="card">
                <div class="card-header">
                    <div class="card-header-left">
                        <h4 class="card-title">
                            <i class="ri-database-2-line text-primary"></i>
                            {{ translate("Enterprise / High Volume Setup") }}
                        </h4>
                    </div>
                </div>
                <div class="card-body">
                    <div class="tip-box">
                        <i class="ri-lightbulb-line"></i>
                        <p>{{ translate("For high-volume sending (100k+ messages/day), use a process manager to run continuous queue workers. This provides maximum performance and auto-restarts on failure.") }}</p>
                    </div>

                    <div class="warning-box mb-4">
                        <i class="ri-information-line"></i>
                        <p>{{ translate("This setup is for Linux servers only. For Windows/macOS high-volume, use the VPS/Local Server setup with multiple terminal windows running queue workers.") }}</p>
                    </div>

                    <ol class="setup-steps">
                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Install Supervisor (Linux)") }}</h5>
                            <p class="step-desc">{{ translate("Connect via SSH and install Supervisor:") }}</p>
                            <div class="code-box">
                                <code>sudo apt-get update && sudo apt-get install supervisor -y</code>
                                <button type="button" class="copy-btn" data-clipboard-text="sudo apt-get update && sudo apt-get install supervisor -y">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Create Worker Configuration") }}</h5>
                            <p class="step-desc">{{ translate("Create a configuration file for xSender:") }}</p>
                            <div class="code-box">
                                <code>sudo nano /etc/supervisor/conf.d/xsender-worker.conf</code>
                                <button type="button" class="copy-btn" data-clipboard-text="sudo nano /etc/supervisor/conf.d/xsender-worker.conf">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Add Configuration Content") }}</h5>
                            <p class="step-desc">{{ translate("Paste this configuration (adjust paths if needed):") }}</p>
                            <div class="code-box">
                                <code>[program:xsender-worker]
process_name=%(program_name)s_%(process_num)02d
command=php {{ base_path('artisan') }} queue:work database --sleep=3 --tries=3 --max-time=3600
directory={{ base_path() }}
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user={{ Arr::get($supervisor ?? [], 'user', 'www-data') }}
numprocs=4
redirect_stderr=true
stdout_logfile={{ base_path('storage/logs/worker.log') }}
stopwaitsecs=3600</code>
                                <button type="button" class="copy-btn" data-clipboard-text="[program:xsender-worker]
process_name=%(program_name)s_%(process_num)02d
command=php {{ base_path('artisan') }} queue:work database --sleep=3 --tries=3 --max-time=3600
directory={{ base_path() }}
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user={{ Arr::get($supervisor ?? [], 'user', 'www-data') }}
numprocs=4
redirect_stderr=true
stdout_logfile={{ base_path('storage/logs/worker.log') }}
stopwaitsecs=3600">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Start Supervisor") }}</h5>
                            <p class="step-desc">{{ translate("Reload and start the workers:") }}</p>
                            <div class="code-box">
                                <code>sudo supervisorctl reread && sudo supervisorctl update && sudo supervisorctl start xsender-worker:*</code>
                                <button type="button" class="copy-btn" data-clipboard-text="sudo supervisorctl reread && sudo supervisorctl update && sudo supervisorctl start xsender-worker:*">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Add Laravel Scheduler") }}</h5>
                            <p class="step-desc">{{ translate("Still need the scheduler for campaigns. Add this cron:") }}</p>
                            <div class="code-box">
                                <code>* * * * * cd {{ base_path() }} && php artisan schedule:run >> /dev/null 2>&1</code>
                                <button type="button" class="copy-btn" data-clipboard-text="* * * * * cd {{ base_path() }} && php artisan schedule:run >> /dev/null 2>&1">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>

                        <li class="setup-step">
                            <h5 class="step-title">{{ translate("Check Status") }}</h5>
                            <p class="step-desc">{{ translate("Verify workers are running:") }}</p>
                            <div class="code-box">
                                <code>sudo supervisorctl status</code>
                                <button type="button" class="copy-btn" data-clipboard-text="sudo supervisorctl status">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                        </li>
                    </ol>

                    <div class="warning-box">
                        <i class="ri-information-line"></i>
                        <p>{{ translate("Increase 'numprocs' value (e.g., 8 or 16) for higher throughput. Each process handles messages independently.") }}</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Test Connection Card (Always visible after selection) -->
        <div class="card mt-4" id="testCard" style="display: none;">
            <div class="card-header">
                <div class="card-header-left">
                    <h4 class="card-title">
                        <i class="ri-flask-line"></i>
                        {{ translate("Test Your Setup") }}
                    </h4>
                </div>
            </div>
            <div class="card-body">
                <p class="text-muted mb-3">{{ translate("After setting up, use these tools to verify everything is working:") }}</p>

                <div class="row g-3">
                    <div class="col-md-6">
                        <div class="border rounded p-3">
                            <h6 class="mb-2"><i class="ri-link text-primary"></i> {{ translate("Automation URL") }}</h6>
                            <p class="small text-muted mb-2">{{ translate("Trigger automation manually:") }}</p>
                            <div class="code-box">
                                <code>{{ url('/automation/run') }}</code>
                                <button type="button" class="copy-btn" data-clipboard-text="{{ url('/automation/run') }}">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                            <a href="{{ url('/automation/run') }}" target="_blank" class="i-btn btn--primary btn--sm mt-2">
                                <i class="ri-external-link-line"></i> {{ translate("Test Now") }}
                            </a>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="border rounded p-3">
                            <h6 class="mb-2"><i class="ri-heart-pulse-line text-success"></i> {{ translate("Health Check URL") }}</h6>
                            <p class="small text-muted mb-2">{{ translate("Check system health:") }}</p>
                            <div class="code-box">
                                <code>{{ url('/automation/health') }}</code>
                                <button type="button" class="copy-btn" data-clipboard-text="{{ url('/automation/health') }}">
                                    <i class="ri-file-copy-line"></i>
                                </button>
                            </div>
                            <a href="{{ url('/automation/health') }}" target="_blank" class="i-btn btn--success btn--sm mt-2">
                                <i class="ri-external-link-line"></i> {{ translate("Check Health") }}
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

    </div>
</main>

<!-- Retry Failed Jobs Confirmation Modal -->
<div class="modal fade actionModal" id="retryFailedModal" tabindex="-1" aria-labelledby="retryFailedModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header text-start">
                <span class="action-icon warning">
                    <i class="ri-restart-line"></i>
                </span>
            </div>
            <div class="modal-body">
                <div class="action-message">
                    <h5>{{ translate("Retry All Failed Jobs?") }}</h5>
                    <p class="text-muted">{{ translate("This will attempt to re-process all failed jobs in the queue. Jobs that failed due to permanent errors may fail again.") }}</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="i-btn btn--dark outline btn--md" data-bs-dismiss="modal">{{ translate("Cancel") }}</button>
                <button type="button" class="i-btn btn--success btn--md" id="confirmRetryFailed">
                    <i class="ri-restart-line"></i> {{ translate("Retry Jobs") }}
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Clear Failed Jobs Confirmation Modal -->
<div class="modal fade actionModal" id="clearFailedModal" tabindex="-1" aria-labelledby="clearFailedModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header text-start">
                <span class="action-icon danger">
                    <i class="bi bi-exclamation-circle"></i>
                </span>
            </div>
            <div class="modal-body">
                <div class="action-message">
                    <h5>{{ translate("Clear All Failed Jobs?") }}</h5>
                    <p class="text-muted">{{ translate("This will permanently delete all failed jobs from the queue. This action cannot be undone.") }}</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="i-btn btn--dark outline btn--md" data-bs-dismiss="modal">{{ translate("Cancel") }}</button>
                <button type="button" class="i-btn btn--danger btn--md" id="confirmClearFailed">
                    <i class="ri-delete-bin-line"></i> {{ translate("Clear Jobs") }}
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Queue Connection Modal -->
<div class="modal fade" id="queueConnectionModal" tabindex="-1" aria-labelledby="queueConnectionModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content">
            <form class="settingsForm" data-route="{{ route('admin.system.setting.store') }}">
                @csrf
                <input type="hidden" name="channel" value="queue">
                <div class="modal-header">
                    <h5 class="modal-title" id="queueConnectionModalLabel">{{ translate("Queue Connection Settings") }}</h5>
                    <button type="button" class="icon-btn btn-ghost btn-sm danger-soft circle modal-closer" data-bs-dismiss="modal">
                        <i class="ri-close-large-line"></i>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="tip-box mb-4">
                        <i class="ri-information-line"></i>
                        <p>
                            <strong>{{ translate("Database") }}</strong> - {{ translate("Best for shared hosting & most users") }}<br>
                            <strong>{{ translate("Redis") }}</strong> - {{ translate("Best for VPS with high volume") }}<br>
                            <strong>{{ translate("Sync") }}</strong> - {{ translate("For testing only (no queue)") }}
                        </p>
                    </div>

                    <div class="row g-4">
                        <div class="col-lg-12">
                            <div class="form-inner">
                                <label for="queue_driver" class="form-label">{{ translate("Queue Driver") }} <span class="text-danger">*</span></label>
                                <select id="queue_driver" name="site_settings[queue_connection_config][driver]" class="form-control select2-search" required>
                                    <option value="database" {{ site_settings('queue_connection_config.driver', 'database') === 'database' ? 'selected' : '' }}>{{ translate("Database (Recommended)") }}</option>
                                    <option value="redis" {{ site_settings('queue_connection_config.driver', 'database') === 'redis' ? 'selected' : '' }}>{{ translate("Redis (High Volume)") }}</option>
                                    <option value="sync" {{ site_settings('queue_connection_config.driver', 'database') === 'sync' ? 'selected' : '' }}>{{ translate("Sync (Testing Only)") }}</option>
                                    <option value="beanstalkd" {{ site_settings('queue_connection_config.driver', 'database') === 'beanstalkd' ? 'selected' : '' }}>{{ translate("Beanstalkd (Advanced)") }}</option>
                                    <option value="sqs" {{ site_settings('queue_connection_config.driver', 'database') === 'sqs' ? 'selected' : '' }}>{{ translate("Amazon SQS (Enterprise)") }}</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-lg-12">
                            <div class="row g-4" id="queue-driver-fields"></div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="i-btn btn--danger outline btn--md" data-bs-dismiss="modal">{{ translate("Close") }}</button>
                    <button type="submit" class="i-btn btn--primary btn--md">{{ translate("Save") }}</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

@push("script-include")
<script src="{{ asset('assets/theme/global/js/select2.min.js') }}"></script>
@endpush

@push("script-push")
<script>
"use strict";

// Server type selection
function selectServerType(type) {
    // Update card selection
    document.querySelectorAll('.server-type-card').forEach(card => {
        card.classList.remove('selected');
    });
    document.querySelector(`[data-server="${type}"]`).classList.add('selected');

    // Show relevant setup instructions
    document.querySelectorAll('.setup-container').forEach(container => {
        container.classList.remove('active');
    });
    document.getElementById(`setup-${type}`).classList.add('active');

    // Show test card
    document.getElementById('testCard').style.display = 'block';

    // Save preference
    localStorage.setItem('xsender_server_type', type);
}

// Copy functionality
function initCopyButtons() {
    document.querySelectorAll('.copy-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.stopPropagation();
            const text = this.dataset.clipboardText || '';

            navigator.clipboard.writeText(text).then(() => {
                notify('success', '{{ translate("Copied!") }}');
                this.innerHTML = '<i class="ri-check-line"></i>';
                setTimeout(() => {
                    this.innerHTML = '<i class="ri-file-copy-line"></i>';
                }, 2000);
            }).catch(() => {
                // Fallback
                const textarea = document.createElement('textarea');
                textarea.value = text;
                textarea.style.position = 'fixed';
                textarea.style.opacity = '0';
                document.body.appendChild(textarea);
                textarea.select();
                document.execCommand('copy');
                document.body.removeChild(textarea);
                notify('success', '{{ translate("Copied!") }}');
            });
        });
    });
}

// Health check
function refreshHealth() {
    const statusIcon = document.getElementById('healthStatusIcon');
    const statusText = document.getElementById('healthStatus');

    statusIcon.innerHTML = '<i class="ri-loader-4-line ri-spin"></i>';
    statusText.textContent = '{{ translate("Checking...") }}';

    fetch('{{ url("/automation/health") }}')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const health = data.data.health;

                // Update status
                if (health.is_healthy) {
                    statusIcon.className = 'health-icon success';
                    statusIcon.innerHTML = '<i class="ri-check-line"></i>';
                    statusText.textContent = '{{ translate("Healthy") }}';
                } else {
                    statusIcon.className = 'health-icon warning';
                    statusIcon.innerHTML = '<i class="ri-alert-line"></i>';
                    statusText.textContent = '{{ translate("Needs Attention") }}';
                }

                // Update values
                document.getElementById('lastCronRun').textContent = health.last_run_ago || '{{ translate("Never") }}';
                document.getElementById('pendingJobs').textContent = health.pending_queue_jobs || '0';
                document.getElementById('failedJobs').textContent = health.failed_queue_jobs || '0';

                // Show warnings
                const warningsDiv = document.getElementById('healthWarnings');
                if (health.warnings && health.warnings.length > 0) {
                    document.getElementById('healthWarningsText').textContent = health.warnings.join('. ');
                    warningsDiv.style.display = 'flex';
                } else {
                    warningsDiv.style.display = 'none';
                }
            }
        })
        .catch(error => {
            statusIcon.className = 'health-icon danger';
            statusIcon.innerHTML = '<i class="ri-error-warning-line"></i>';
            statusText.textContent = '{{ translate("Error") }}';
        });
}

function retryFailed() {
    const btn = document.getElementById('confirmRetryFailed');
    const originalHtml = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span> {{ translate("Processing...") }}';

    fetch('{{ url("/automation/retry-failed") }}')
        .then(response => response.json())
        .then(data => {
            notify(data.success ? 'success' : 'error', data.message);
            if (data.success) refreshHealth();
            bootstrap.Modal.getInstance(document.getElementById('retryFailedModal')).hide();
        })
        .catch(() => notify('error', '{{ translate("Request failed") }}'))
        .finally(() => {
            btn.disabled = false;
            btn.innerHTML = originalHtml;
        });
}

function clearFailed() {
    const btn = document.getElementById('confirmClearFailed');
    const originalHtml = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span> {{ translate("Processing...") }}';

    fetch('{{ url("/automation/clear-failed") }}')
        .then(response => response.json())
        .then(data => {
            notify(data.success ? 'success' : 'error', data.message);
            if (data.success) refreshHealth();
            bootstrap.Modal.getInstance(document.getElementById('clearFailedModal')).hide();
        })
        .catch(() => notify('error', '{{ translate("Request failed") }}'))
        .finally(() => {
            btn.disabled = false;
            btn.innerHTML = originalHtml;
        });
}

// Queue driver fields
window.connections = @json($connections ?? []);

// Mode descriptions
const modeDescriptions = {
    'auto': '{{ translate("Auto Detect: System automatically detects supervisor workers and adjusts behavior. Best for most users.") }}',
    'cron_url': '{{ translate("Cron URL: The /automation/run URL handles both campaigns AND queue processing. Use this for cPanel/shared hosting where you set up a single cron job.") }}',
    'scheduler': '{{ translate("Laravel Scheduler: artisan schedule:run handles campaigns AND queue processing. Use this for VPS where you run the scheduler cron.") }}',
    'supervisor': '{{ translate("Supervisor Workers: Supervisor handles queue processing continuously. Scheduler only handles campaign scheduling. Best for high-volume enterprise setups.") }}'
};

// Update mode description on change
$('#automationModeSelect').on('change', function() {
    const mode = $(this).val();
    $('#modeDescriptionText').text(modeDescriptions[mode] || modeDescriptions['auto']);
});

// Save automation mode
$('#automationModeForm').on('submit', function(e) {
    e.preventDefault();
    const mode = $('#automationModeSelect').val();
    const btn = $(this).find('button[type="submit"]');
    const originalHtml = btn.html();

    btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-1"></span> {{ translate("Saving...") }}');

    $.ajax({
        url: '{{ route("admin.system.automation.mode") }}',
        method: 'POST',
        data: {
            _token: '{{ csrf_token() }}',
            mode: mode
        },
        success: function(response) {
            if (response.success) {
                notify('success', response.message || '{{ translate("Automation mode saved successfully") }}');
                // Update the badge
                $('.card-header-right .badge').text('{{ translate("Current:") }} ' + mode.replace('_', ' '));
            } else {
                notify('error', response.message || '{{ translate("Failed to save automation mode") }}');
            }
        },
        error: function() {
            notify('error', '{{ translate("Failed to save automation mode") }}');
        },
        complete: function() {
            btn.prop('disabled', false).html(originalHtml);
        }
    });
});

$(document).ready(function() {
    initCopyButtons();
    refreshHealth();

    // Trigger mode description update on load
    $('#automationModeSelect').trigger('change');

    // Restore server type selection
    const savedType = localStorage.getItem('xsender_server_type');
    if (savedType) {
        selectServerType(savedType);
    }

    // Modal confirm button handlers
    $('#confirmRetryFailed').on('click', function() {
        retryFailed();
    });

    $('#confirmClearFailed').on('click', function() {
        clearFailed();
    });

    // Select2
    if ($('.select2-search').length) {
        select2_search($('.select2-search').data('placeholder'), "#queueConnectionModal");
    }

    // Queue driver change
    $('#queue_driver').on('change', function() {
        const driver = $(this).val();
        const $fields = $('#queue-driver-fields');
        $fields.empty();

        const config = window.connections[driver] || {};
        if (Object.keys(config).length === 0) {
            $fields.append('<div class="col-12"><p class="text-muted small">{{ translate("No additional configuration required.") }}</p></div>');
            return;
        }

        Object.entries(config).forEach(([key, cfg]) => {
            const type = ['password', 'secret'].includes(key) ? 'password' : 'text';
            const required = cfg.required ? '<span class="text-danger">*</span>' : '';
            $fields.append(`
                <div class="col-lg-6">
                    <div class="form-inner">
                        <label class="form-label">${cfg.label}${required}</label>
                        <input type="${type}" name="site_settings[queue_connection_config][connection][${key}]"
                               class="form-control" placeholder="${cfg.placeholder}"
                               value="${cfg.value || ''}" ${cfg.required ? 'required' : ''}>
                    </div>
                </div>
            `);
        });
    });

    $('#queueConnectionModal').on('show.bs.modal', function() {
        const driver = window.connections.driver || 'database';
        $('#queue_driver').val(driver).trigger('change');
    });
});
</script>
@endpush
