import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

/**
 * Runtime Configuration
 * Values set dynamically from Laravel via API
 * Now persists to file to survive restarts
 */

class RuntimeConfig {
  constructor() {
    this.configPath = path.resolve(__dirname, '../../storage/.runtime-config.json');

    // Default values (can be overridden by Laravel)
    this.config = {
      apiKey: process.env.API_KEY || 'default-change-me',
      allowedOrigins: process.env.ALLOWED_ORIGINS?.split(',') || ['*'],
      purchaseKey: process.env.PURCHASE_KEY || '',
      envatoUsername: process.env.ENVATO_USERNAME || '',
      softwareId: process.env.SOFTWARE_ID || 'BX32DOTW4Q797ZF3',
      version: process.env.VERSION || '3.3',
      domain: process.env.DOMAIN || '',
      laravelConfigured: false,
    };

    // Load persisted config on startup
    this._loadPersistedConfig();
  }

  /**
   * Load configuration from persisted file
   * This ensures config survives PM2 restarts
   */
  _loadPersistedConfig() {
    try {
      if (fs.existsSync(this.configPath)) {
        const saved = JSON.parse(fs.readFileSync(this.configPath, 'utf8'));

        // Only restore if saved config is recent (within 7 days)
        if (saved.savedAt && Date.now() - saved.savedAt < 7 * 24 * 60 * 60 * 1000) {
          // Restore saved values, but keep env values for apiKey as fallback
          if (saved.apiKey) this.config.apiKey = saved.apiKey;
          if (saved.allowedOrigins) this.config.allowedOrigins = saved.allowedOrigins;
          if (saved.purchaseKey) this.config.purchaseKey = saved.purchaseKey;
          if (saved.envatoUsername) this.config.envatoUsername = saved.envatoUsername;
          if (saved.softwareId) this.config.softwareId = saved.softwareId;
          if (saved.version) this.config.version = saved.version;
          if (saved.domain) this.config.domain = saved.domain;

          this.config.laravelConfigured = true;
          console.log('[RuntimeConfig] Restored configuration from persistent storage');
        }
      }
    } catch (error) {
      console.error('[RuntimeConfig] Failed to load persisted config:', error.message);
    }
  }

  /**
   * Save configuration to persistent storage
   */
  _persistConfig() {
    try {
      const dir = path.dirname(this.configPath);
      if (!fs.existsSync(dir)) {
        fs.mkdirSync(dir, { recursive: true });
      }

      const dataToSave = {
        apiKey: this.config.apiKey,
        allowedOrigins: this.config.allowedOrigins,
        purchaseKey: this.config.purchaseKey,
        envatoUsername: this.config.envatoUsername,
        softwareId: this.config.softwareId,
        version: this.config.version,
        domain: this.config.domain,
        savedAt: Date.now()
      };

      fs.writeFileSync(this.configPath, JSON.stringify(dataToSave, null, 2));
      console.log('[RuntimeConfig] Configuration persisted to storage');
    } catch (error) {
      console.error('[RuntimeConfig] Failed to persist config:', error.message);
    }
  }

  /**
   * Update configuration from Laravel
   */
  updateFromLaravel(laravelConfig) {
    if (laravelConfig.apiKey) {
      this.config.apiKey = laravelConfig.apiKey;
    }

    if (laravelConfig.allowedOrigins) {
      this.config.allowedOrigins = Array.isArray(laravelConfig.allowedOrigins)
        ? laravelConfig.allowedOrigins
        : [laravelConfig.allowedOrigins];
    }

    if (laravelConfig.purchaseKey) {
      this.config.purchaseKey = laravelConfig.purchaseKey;
    }

    if (laravelConfig.envatoUsername) {
      this.config.envatoUsername = laravelConfig.envatoUsername;
    }

    if (laravelConfig.softwareId) {
      this.config.softwareId = laravelConfig.softwareId;
    }

    if (laravelConfig.version) {
      this.config.version = laravelConfig.version;
    }

    if (laravelConfig.domain) {
      this.config.domain = laravelConfig.domain;
    }

    this.config.laravelConfigured = true;

    // Persist config to file so it survives restarts
    this._persistConfig();
  }

  /**
   * Get configuration value
   */
  get(key) {
    return this.config[key];
  }

  /**
   * Check if configured by Laravel
   */
  isConfigured() {
    return this.config.laravelConfigured;
  }

  /**
   * Get all config
   */
  getAll() {
    return { ...this.config };
  }

  /**
   * Clear persisted config (useful for troubleshooting)
   */
  clearPersisted() {
    try {
      if (fs.existsSync(this.configPath)) {
        fs.unlinkSync(this.configPath);
        console.log('[RuntimeConfig] Persisted config cleared');
      }
    } catch (error) {
      console.error('[RuntimeConfig] Failed to clear persisted config:', error.message);
    }
  }
}

// Export singleton instance
export default new RuntimeConfig();
