<?php

namespace App\Http\Middleware;

use Closure;
use App\Models\Language;
use App\Enums\StatusEnum;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;

class LanguageMiddleware
{
    /**
     * Handle an incoming request.
     * Uses caching to avoid database queries on every request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        try {
            $lang_code = $request->header('X-App-Language') ?? session('locale');

            // If we have a locale in session, use it without DB query
            if ($lang_code && session('locale') === $lang_code) {
                App::setLocale($lang_code);
                return $next($request);
            }

            // Get available languages from cache (avoid DB query)
            $languages = $this->getCachedLanguages();

            if ($lang_code && isset($languages[$lang_code])) {
                App::setLocale($lang_code);
                session(['locale' => $lang_code]);
                return $next($request);
            }

            // Get default language from cache
            $defaultLang = $this->getDefaultLanguage($languages);
            App::setLocale($defaultLang);
            session(['locale' => $defaultLang]);

        } catch (\Exception $e) {
            // Database not available (fresh install) - use default locale
            App::setLocale('en');
        }

        return $next($request);
    }

    /**
     * Get languages from cache
     */
    private function getCachedLanguages(): array
    {
        return Cache::remember('available_languages', now()->addHours(24), function () {
            try {
                return Language::where('status', StatusEnum::TRUE->status())
                    ->pluck('code', 'code')
                    ->toArray();
            } catch (\Exception $e) {
                return ['en' => 'en'];
            }
        });
    }

    /**
     * Get default language code
     */
    private function getDefaultLanguage(array $languages): string
    {
        static $defaultLang = null;

        if ($defaultLang !== null) {
            return $defaultLang;
        }

        $defaultLang = Cache::remember('default_language', now()->addHours(24), function () {
            try {
                return Language::where('is_default', StatusEnum::TRUE->status())
                    ->value('code') ?? 'en';
            } catch (\Exception $e) {
                return 'en';
            }
        });

        return $defaultLang;
    }
}
